from axelrod.action import Action, actions_to_str

from axelrod.player import Player

from axelrod.strategy_transformers import (
    FinalTransformer,
    TrackHistoryTransformer,
)

C, D = Action.C, Action.D

class OriginalGradual(Player):
    """
    A player that punishes defections with a growing number of defections
    but after punishing for `punishment_limit` number of times enters a calming
    state and cooperates no matter what the opponent does for two rounds.

    The `punishment_limit` is incremented whenever the opponent defects and the
    strategy is not in either calming or punishing state.

    Note that `Gradual` appears in [CRISTAL-SMAC2018]_ however that version of
    `Gradual` does not give the results reported in [Beaufils1997]_ which is the
    paper that first introduced the strategy. For a longer discussion of this
    see: https://github.com/Axelrod-Python/Axelrod/issues/1294. This is why this
    strategy has been renamed to `OriginalGradual`.

    Names:

    - Gradual: [Beaufils1997]_
    """

    name = "Original Gradual"
    classifier = {
        "memory_depth": float("inf"),
        "stochastic": False,
        "long_run_time": False,
        "inspects_source": False,
        "manipulates_source": False,
        "manipulates_state": False,
    }

    def __init__(self) -> None:

        super().__init__()
        self.calming = False
        self.punishing = False
        self.punishment_count = 0
        self.punishment_limit = 0

    def strategy(self, opponent: Player) -> Action:
        """Actual strategy definition that determines player's action."""

        if self.calming:
            self.calming = False
            return C

        if self.punishing:
            if self.punishment_count < self.punishment_limit:
                self.punishment_count += 1
                return D
            else:
                self.calming = True
                self.punishing = False
                self.punishment_count = 0
                return C

        if D in opponent.history[-1:]:
            self.punishing = True
            self.punishment_count += 1
            self.punishment_limit += 1
            return D

        return C